﻿using Pervasive.Data.SqlClient;
using System;
using System.Data;
using System.IO;
using System.Windows.Forms;
using lib;
using System.Drawing;
using System.Xml;
using BTServerDB.lib;
using Newtonsoft.Json;
using System.Collections.Generic;

namespace BTServerDBDotNetFrameWork
{
    public partial class Form1 : Form
    {
        const String _DEF_LISTEN_PORT = "11433";
        public const String _SW_NAME = "BTServerLib - v.";
        public const String _SW_VERSION = "1.0.0";
        protected LogWriter logger = LogWriter.getInstance();

        bool isListen = false;
        AsynchronousSocketListener socket;

        public Form1()
        {
            InitializeComponent();

            socket = new AsynchronousSocketListener(this);
            socket.OnReceive += Socket_OnReceive;
            socket.OnStateChanged += Socket_OnStateChanged;
        }

        private void Socket_OnStateChanged(ConnectionState state)
        {
            Color color = Color.Red;
            switch (state)
            {
                case ConnectionState.Listen:
                    color = Color.Green;
                    logger.Info("TCP server start listen on port " + _DEF_LISTEN_PORT);
                    break;
                case ConnectionState.Accepted:
                    color = Color.Orange;
                    logger.Info("Connection established.");
                    break;
                case ConnectionState.Busy:
                    color = Color.Blue;
                    break;
                case ConnectionState.Stop:
                    color = Color.Red;
                    logger.Warn("TCP server stopped.");
                    break;
            }
        }

        private String Socket_OnReceive(string content)
        {
            logger.Debug("Received : " + content);
            String msg = "{\"status\":\"ERROR\", \"error\":\"INVALID COMMAND\"}";

            try
            {
                JsonBtCommand json = JsonConvert.DeserializeObject<JsonBtCommand>(content);
                String conn_str = json.conn_str.ToString();

                switch (json.cmd)
                {
                    case "QUERY":
                        List<Dictionary<String, String>> data = ExecuteQuery(conn_str, json.sql);

                        JsonResponseQuery jsonResponseQuery = new JsonResponseQuery()
                        {
                            cmd = "QUERY_ACK",
                            status = "SUCCESS",
                            error = "",
                            data = data
                        };
                        msg = JsonConvert.SerializeObject(jsonResponseQuery, Newtonsoft.Json.Formatting.Indented);
                        break;

                    case "EXECUTE":
                        int affected_rows = ExecuteNonQuery(conn_str, json.sql);
                        JsonResponseExecute jsonResponse = new JsonResponseExecute()
                        {
                            cmd = "EXECUTE_ACK",
                            status = affected_rows > 0 ? "SUCCESS" : "ERROR",
                            affected_rows = affected_rows,
                            error = ""
                        };
                        msg = JsonConvert.SerializeObject(jsonResponse, Newtonsoft.Json.Formatting.Indented);
                        break;

                    default:
                        logger.Warn("Invalid command " + json.cmd);
                        break;

                }

            }
            catch (Exception e)
            {
                logger.Log(e.ToString());
            }

            return msg + "\n\n\n";
        }

        private void btnTestDb_Click(object sender, EventArgs e)
        {
            String conn_str = "Host=localhost;Port=1583;Database=DEMODATA;";
            String sql = "SELECT * FROM Billing";

            StringWriter writer = new StringWriter();
            ExecuteQuery(conn_str, sql, writer);
            logger.Debug(writer.ToString());

            sql = "UPDATE Billing SET Comments='test5' WHERE Student_ID=100062607";
            ExecuteNonQuery(conn_str, sql);
        }


        // This method call dumps the StringWriter contents on to the ResultSet window
        private string flush(StringWriter writer)
        {
            string buffer = writer.ToString();
            logger.Debug(buffer);
            writer.GetStringBuilder().Length = 0;

            return buffer;
        }

        // This method call will print out the result set on to the ResultSet window
        public void printReader(IDataReader reader, StringWriter writer)
        {
            bool fMoreRows;
            int i;
            int length = 0;
            bool fFirst = true;
            for (i = 0; i < reader.FieldCount; ++i)
            {
                if (!fFirst)
                {
                    writer.Write(", ");
                }
                fFirst = false;
                writer.Write(reader.GetName(i));
                length = length + (reader.GetName(i).Length);
            }
            writer.WriteLine("");
            for (int z = 0; z < length + ((reader.FieldCount) * 2); z++)
            {
                writer.Write("-");
            }
            writer.WriteLine("");
            do
            {
                fMoreRows = reader.Read();
                if (fMoreRows)
                {
                    fFirst = true;
                    for (i = 0; i < reader.FieldCount; ++i)
                    {
                        if (!fFirst)
                        {
                            writer.Write(", ");
                        }
                        fFirst = false;

                        if (reader.IsDBNull(i))
                        {
                            writer.Write("<null>");
                        }
                        else
                        {
                            string str;
                            object obj = reader.GetValue(i);
                            if (obj is DateTime)
                            {
                                str = ((DateTime)obj).ToString("yyyy-MM-dd HH:mm:ss.fff");
                            }
                            else
                            {
                                str = obj.ToString();
                            }

                            writer.Write(str);
                        }
                    }
                    writer.WriteLine("");
                }
            } while (fMoreRows);
        }

        // This method call will print out the result set on to the ResultSet window
        public void printReaderJson(IDataReader reader, StringWriter writer)
        {
            bool fMoreRows;
            int i;
            bool fFirst = true;
            bool fFirstRow = true;
            do
            {
                fMoreRows = reader.Read();
                if (fMoreRows)
                {
                    if (!fFirstRow)
                    {
                        writer.Write(",");
                    }
                    writer.Write("{");

                    fFirstRow = false;
                    fFirst = true;
                    for (i = 0; i < reader.FieldCount; ++i)
                    {
                        if (!fFirst)
                        {
                            writer.Write(", ");
                        }
                        fFirst = false;

                        if (reader.IsDBNull(i))
                        {
                            writer.Write("\"{0}\":<null>", reader.GetName(i));
                        }
                        else
                        {
                            string str;
                            object obj = reader.GetValue(i);
                            if (obj is DateTime)
                            {
                                str = ((DateTime)obj).ToString("yyyy-MM-dd HH:mm:ss.fff");
                            }
                            else
                            {
                                str = obj.ToString();
                            }

                            writer.Write("\"{0}\":\"{1}\"", reader.GetName(i), str);
                        }
                    }

                    writer.WriteLine("}");
                }
            } while (fMoreRows);
        }

        // This method call will print out the result set on to the ResultSet window
        public List<Dictionary<string, string>> printReaderJsonObject(IDataReader reader)
        {
            List<Dictionary<string, string>> data = new List<Dictionary<string, string>>();

            bool fMoreRows;
            int i;
            do
            {
                fMoreRows = reader.Read();
                if (fMoreRows)
                {
                    Dictionary<string, string> cols = new Dictionary<string, string>();

                    for (i = 0; i < reader.FieldCount; ++i)
                    {
                        if (reader.IsDBNull(i))
                        {
                            cols.Add(reader.GetName(i), "<null>");
                        }
                        else
                        {
                            string str;
                            object obj = reader.GetValue(i);
                            if (obj is DateTime)
                            {
                                str = ((DateTime)obj).ToString("yyyy-MM-dd HH:mm:ss.fff");
                            }
                            else
                            {
                                str = obj.ToString();
                            }

                            cols.Add(reader.GetName(i), str);
                        }
                    }

                    data.Add(cols);
                }
            } while (fMoreRows);

            return data;
        }

        private void ExecuteQuery(String conn_str, String sql, StringWriter writer)
        {
            logger.Debug("Connection string => " + conn_str);
            logger.Debug("SQL => " + sql);

            // Modify the connection string to match your servers
            PsqlConnection Conn = new PsqlConnection(conn_str);
            try
            {
                Conn.Open();
            }
            catch (Exception ex)
            {
                // Connection failed
                logger.Error(ex.Message);
                logger.Error(ex.StackTrace);
                return;
            }
            // PsqlCommand
            PsqlCommand DBCmd = new PsqlCommand(sql, Conn);
            // Call PsqlDataReader
            try
            {
                PsqlDataReader dataReader = DBCmd.ExecuteReader();
                // Print out the result set onto the ResultSet window
                //printReaderJson(dataReader, writer);
                printReaderJson(dataReader, writer);
                // Close the DataReader
                dataReader.Close();
            }
            catch (Exception ex)
            {
                // Exception thrown at execution
                logger.Error(ex.Message);
                logger.Error(ex.StackTrace);
            }
            // Close the connection
            Conn.Close();
        }

        private List<Dictionary<string, string>> ExecuteQuery(String conn_str, String sql)
        {
            List<Dictionary<string, string>> data = null;
            logger.Debug("Connection string => " + conn_str);
            logger.Debug("SQL => " + sql);

            // Modify the connection string to match your servers
            PsqlConnection Conn = new PsqlConnection(conn_str);
            try
            {
                Conn.Open();
            }
            catch (Exception ex)
            {
                // Connection failed
                logger.Error(ex.Message);
                logger.Error(ex.StackTrace);
                return null;
            }
            // PsqlCommand
            PsqlCommand DBCmd = new PsqlCommand(sql, Conn);
            // Call PsqlDataReader
            try
            {
                PsqlDataReader dataReader = DBCmd.ExecuteReader();
                // Print out the result set onto the ResultSet window
                //printReaderJson(dataReader, writer);
                data = printReaderJsonObject(dataReader);
                // Close the DataReader
                dataReader.Close();
            }
            catch (Exception ex)
            {
                // Exception thrown at execution
                logger.Error(ex.Message);
                logger.Error(ex.StackTrace);
            }
            // Close the connection
            Conn.Close();

            return data;
        }

        private int ExecuteNonQuery(String conn_str, String sql)
        {
            int affected_rows = 0;

            logger.Debug("Connection string => " + conn_str);
            logger.Debug("SQL => " + sql);

            // Modify the connection string to match your servers
            PsqlConnection Conn = new PsqlConnection(conn_str);
            try
            {
                Conn.Open();
            }
            catch (Exception ex)
            {
                // Connection failed
                logger.Error(ex.Message);
                return -1;
            }
            // PsqlCommand
            PsqlCommand DBCmd = new PsqlCommand(sql, Conn);
            // Call PsqlDataReader
            try
            {
                affected_rows = DBCmd.ExecuteNonQuery();
            }
            catch (Exception ex)
            {
                // Exception thrown at execution
                logger.Error(ex.Message);
            }
            // Close the connection
            Conn.Close();

            return affected_rows;
        }

        private async void btnStartStopServer_Click(object sender, EventArgs e)
        {
            if (isListen)
            {
                socket.StopListening();
            }
            else
            {
                await socket.StartListening(Int32.Parse(_DEF_LISTEN_PORT));
            }
            isListen = !isListen;
            btnStartStopServer.Text = isListen ? "Stop" : "Start";

        }

        private void Form1_FormClosed(object sender, FormClosedEventArgs e)
        {
            if (isListen)
                socket.StopListening();

        }
    }
}
